/*____________________________________________________________________________
	Copyright (C) 2000 Networks Associates Technology, Inc.
	All rights reserved.
	
	Platform independent implementation of SECSH v 1.5
	
	$Id: pgpSECSH.c,v 1.6 2001/03/20 00:05:15 hal Exp $
____________________________________________________________________________*/

#include "pgpSECSHPriv.h"

#include "pgpErrors.h"
#include "pgpContext.h"
#include "pgpMem.h"
#include "pgpEndianConversion.h"
#include "pgpHash.h"
#include "pgpHMAC.h"
#include "pgpPublicKey.h"
#include "pgpSymmetricCipher.h"
#include "pgpCBC.h"
#include "pgpCFB.h"
#include "pgpKeys.h"
#include "pgpFeatures.h"
#include "pgpPFLPriv.h"
#include "pgpKeyPriv.h"
#include "pgpBigNum.h"

#include "pgpOptionListPriv.h"

#include <string.h>


#define CKERR			if( err ) goto done	
#define CKNULL(p)		if( IsNull(p) ) {					\
							err = kPGPError_OutOfMemory;	\
							goto done; }
#define FATALSECSH( x )	{									\
			err = x;										\
			session->state = kPGPsecsh_FatalErrorState;		\
			goto done;	}
#define CRC32_POLY		0xedb88320L
#define SECSHIDPREFIX	"SSH-"
#define	SECSHID			"SSH-1.4-PGPTest1.0\n"
#define SECSHNUMCLIENTSTATES	5
#define SECSHNUMEVENTS			5

/* -1 is ignored, -2 is an error */
static PGPInt8 kPGPsecsh_ClientStateTable[SECSHNUMCLIENTSTATES][SECSHNUMEVENTS] =
{
	/*		 S	 F	PK	CH	 D	*/
{	/* 0 */	-2,	-2,	 1,	-2,	 0,	},		/* Waiting for Server PK */
{	/* 1 */	 2,	-2,	-2,	-2,	 1,	},		/* Sent session key */
{	/* 2 */	 5,	 3,	-2,	-2,	 2,	},		/* Sent user name */
{	/* 3 */	-2,	-2,	-2,	 4,	 3,	},		/* Requested RSA challenge */
{	/* 4 */	 5,	-2,	-2,	-2,	 4,	}		/* Sent RSA response */
};


	PGPInt8
pgpSECSHNextState(
	PGPsecshSessionPriv *	session,
	PGPByte					event )
{
	PGPInt8					newState;
	
	pgpAssert( session->isClientSide );
	newState = kPGPsecsh_ClientStateTable[session->intState][event];
	if( newState == -2 )
		session->state = kPGPsecsh_FatalErrorState;
	else if( newState >= 0 )
		session->intState = newState;
	return newState;
}



	static void
sCalculateCRC(
	PGPByte const *		inBuffer,
	PGPSize				inLength,
	PGPByte *			outCRC )
{
	PGPUInt32 crc = 0;
	PGPUInt32 b;
	PGPUInt32 i;

	while( inLength-- )
	{
		crc ^= *inBuffer++;
		for( i = 0; i < 8; i++ )
		{
			b = (crc & 1);
			crc >>= 1;
			if( b )
				crc ^= CRC32_POLY;
		}
	}
	PGPUInt32ToEndian( crc, kPGPBigEndian, outCRC );
}


/* Return the size in bits of the bignum stored in bigendian form in
 * inBuffer.  Return outOffset as the offset to the first nonzero
 * byte.
 */
	static PGPUInt16
sCountBits(
	const PGPByte *		inBuffer,
	PGPSize				inLength,
	PGPSize *			outOffset )
{
	PGPSize				off;
	PGPUInt16			cnt;

	for( off=0; off<inLength; ++off )
		if( inBuffer[off] != 0 )
			break;
	*outOffset = off;
	if( off == inLength )
		return 0;
	for( cnt=7; (inBuffer[off] & (1<<cnt))==0; --cnt )
		;
	return cnt + 8*(inLength - off - 1);
}

/*
 * Starting with RSA key data in SECSH format (exponent then modulus),
 * create a PGP key with that data and return the keydb and key.
 */
	static PGPError
sDataToPGPKey (
	PGPsecshSessionPriv *	session,
	PGPByte *				inBuffer,
	PGPUInt32				inLength,
	PGPUInt32				*outDataUsed,
	PGPKeyDBRef *			outKDB,
	PGPKeyDBObjRef *		outKey)
{
	PGPUInt32				expbytes,
							modbytes;
	PGPUInt32				keybufLength;
	PGPUInt32				len = inLength;
	PGPByte *				p = inBuffer;
	PGPByte					*bp,
							*exponent,
							*modulus;
	PGPKeyIterRef			kiter;
	PGPKeyDBRef				kdb;
	PGPKeyDBObjRef			key;
	PGPByte *				pgpBuffer = NULL;
	PGPSize					bufLength = 0;
	PGPError				err = kPGPError_NoErr;

	*outDataUsed = 0;
	*outKDB = kInvalidPGPKeyDBRef;
	*outKey = kInvalidPGPKeyDBObjRef;

	expbytes = (((p[0]<<8) | p[1]) + 7) / 8;
	if( len < expbytes + 2 )
	{
		(void)pgpSECSHAlert( session,	kPGPsecsh_AL_FatalAlert,
									kPGPsecsh_AT_DecodeError );
		FATALSECSH( kPGPError_SECSHProtocolViolation );
	}
	exponent = p;
	p += expbytes + 2;
	len -= expbytes + 2;
	modbytes = (((p[0]<<8) | p[1]) + 7) / 8;
	if( len < modbytes + 2 )
	{
		(void)pgpSECSHAlert( session,	kPGPsecsh_AL_FatalAlert,
									kPGPsecsh_AT_DecodeError );
		FATALSECSH( kPGPError_SECSHProtocolViolation );
	}
	modulus = p;
	p += modbytes + 2;
	len -= modbytes + 2;

	keybufLength = 1 + 2 + 1 + 4 + 2 + 1 + 2 + modbytes + 2 + expbytes;
	bufLength = keybufLength + 2 + strlen( session->hostName );
	pgpBuffer = (PGPByte *) PGPNewData(session->memMgr, bufLength, 0 );
	CKNULL( pgpBuffer );
	bp = pgpBuffer;
	*bp++ = 0x99; /* RSA key buffer */
	*bp++ = ((keybufLength-3)>>8) & 0xff;
	*bp++ = (keybufLength-3) & 0xff;
	*bp++ = 0x03; /* key version */
	*bp++ = 0x00; /* creation date */
	*bp++ = 0x00;
	*bp++ = 0x00;
	*bp++ = 0x00;
	*bp++ = 0x00; /* expiration interval */
	*bp++ = 0x00;
	*bp++ = 0x01; /* key type (1 = RSA) */
	pgpCopyMemory( modulus, bp, modbytes+2 );
	bp += modbytes+2;
	pgpCopyMemory( exponent, bp, expbytes+2 );
	bp += expbytes+2;

	*bp++ = 0xb4; /* userid */
	*bp++ = strlen( session->hostName );
	pgpCopyMemory( session->hostName, bp, strlen( session->hostName ) );
	bp += strlen( session->hostName );

	PGPImport( session->pgpContext, &kdb,
			   PGPOInputBuffer( session->pgpContext, pgpBuffer, bufLength ),
			   PGPOLastOption( session->pgpContext ) );
	if( !PGPKeyDBRefIsValid( kdb ) )
	{
		(void)pgpSECSHAlert( session,	kPGPsecsh_AL_FatalAlert,
									kPGPsecsh_AT_DecodeError );
		FATALSECSH( kPGPError_SECSHProtocolViolation );
	}

	PGPNewKeyIterFromKeyDB( kdb, &kiter );
	PGPKeyIterNextKeyDBObj( kiter, kPGPKeyDBObjType_Key, &key );
	PGPFreeKeyIter( kiter );

	*outDataUsed = p - inBuffer;
	*outKDB = kdb;
	*outKey = key;

done:
	if( IsntNull( pgpBuffer ) )
	{
		(void)PGPFreeData( pgpBuffer );
	}
	return err;
}

/* Print a bignum to a buffer, return length of string.  If buffer is
 * null just return the length. */
	static PGPInt32
sPrint10(PGPContextRef context, char *outbuf, PGPBigNumRef bn)
{
	PGPBigNumRef pbig, pbig1;
	PGPBigNumRef ten, zero, rem;
	char buf[3000];			/* up to 9000 bits */
	PGPInt32 bufi = sizeof(buf)-1;
	PGPInt32 n;

	buf[bufi] = '\0';
	PGPNewBigNum( context, FALSE, &pbig );
	PGPNewBigNum( context, FALSE, &pbig1 );
	PGPNewBigNum( context, FALSE, &rem );
	
	PGPNewBigNum( context, FALSE, &zero );
	PGPNewBigNum( context, FALSE, &ten );
	PGPBigNumSetQ( ten, 10 );

	PGPAssignBigNum (bn, pbig);
	
	while (PGPBigNumCompare (pbig, zero) != 0) {
		PGPBigNumDivide( pbig, ten, pbig1, rem );
		PGPAssignBigNum( pbig1, pbig );
		n = PGPBigNumGetLSWord (rem);
		buf[--bufi] = n + '0';
	}

	PGPFreeBigNum (zero);
	PGPFreeBigNum (ten);
	PGPFreeBigNum (pbig);
	PGPFreeBigNum (pbig1);
	PGPFreeBigNum (rem);

	if( outbuf != NULL )
		memcpy( outbuf, buf+bufi, sizeof(buf)-bufi );

	return sizeof(buf) - bufi - 1;
}



	PGPError
pgpSECSHClientChooseAlgorithm(
	PGPsecshSessionPriv *	session,
	PGPCipherAlgorithm *	outCipher )
{
	PGPError				err = kPGPError_NoErr;

	if( ! ( session->cipherMask & (1 << kPGPsecsh_CT_IDEA) ) )
	{
		(void)pgpSECSHAlert( session,	kPGPsecsh_AL_FatalAlert,
									kPGPsecsh_AT_InsufficientSecurity );
		FATALSECSH( kPGPError_SECSHNoCommonCipher );
	}

	*outCipher = kPGPCipherAlgorithm_IDEA;

 done:
	return err;
}


	PGPError
pgpSECSHBufferRawData(
	PGPsecshSessionPriv *	session,
	const PGPByte *			rawData,
	PGPSize					rawDataSize )
{
	PGPError				err = kPGPError_NoErr;
	
	err = PGPReallocData( session->memMgr,
							(void **) &session->rcvdRawData,
							session->rawDataSize + rawDataSize, 0 ); CKERR;
	pgpCopyMemory( rawData, session->rcvdRawData + session->rawDataSize,
					rawDataSize );
	session->rawDataSize += rawDataSize;

done:
	return err;
}


	PGPError
pgpSECSHExtractRawData(
	PGPsecshSessionPriv *	session,
	PGPByte *				rawData,
	PGPSize *				rawDataSize )
{
	PGPError				err = kPGPError_NoErr;
	PGPSize					maxSize = *rawDataSize;
	
	*rawDataSize = 0;
	if( ( maxSize > 0 ) && ( session->rawDataSize > 0 ) )
	{
		if( maxSize > session->rawDataSize )
			maxSize = session->rawDataSize;
		pgpCopyMemory( session->rcvdRawData, rawData, maxSize );
		*rawDataSize = maxSize;
		pgpCopyMemory( session->rcvdRawData + maxSize,
						session->rcvdRawData, session->rawDataSize - maxSize );
		session->rawDataSize -= maxSize;
		err = PGPReallocData( session->memMgr,
							(void **) &session->rcvdRawData,
							session->rawDataSize, 0 ); CKERR;
	}
done:
	return err;
}

	PGPError
pgpSECSHBufferSendData(
	PGPsecshSessionPriv *	session,
	const PGPByte *			sendData,
	PGPSize					sendDataSize )
{
	PGPError				err = kPGPError_NoErr;
	
	err = PGPReallocData( session->memMgr,
					(void **) &session->queuedSendData,
					session->queuedSendSize + sendDataSize, 0 ); CKERR;
	pgpCopyMemory( sendData, session->queuedSendData + session->queuedSendSize,
					sendDataSize );
	session->queuedSendSize += sendDataSize;

done:
	return err;
}


	PGPError
pgpSECSHReceivePacket(
	PGPsecshSessionPriv *	session,
	PGPByte *				outType,
	PGPByte **				outBuffer,
	PGPSize *				outLength )
{
	PGPError				err	= kPGPError_NoErr;
    PGPError            	rerr = kPGPError_NoErr;
	PGPByte					header[kPGPsecsh_RecordLengthSize];
	PGPByte *				buffer = NULL;
	PGPSize					rcvd;
	PGPUInt16				length;
	PGPUInt16				bufLength;
	PGPUInt8				padLength;
	PGPInt32				bytesRead;
	PGPByte					crcCheck[kPGPsecsh_CRCSize];

	*outType	= 0;
	*outBuffer	= NULL;
	*outLength	= 0;
	
	rcvd = kPGPsecsh_RecordLengthSize;
	err = pgpSECSHExtractRawData( session, header, &rcvd );	CKERR;
	for(; rcvd < kPGPsecsh_RecordLengthSize; rcvd += bytesRead)
	{
		bytesRead = (session->secshReceiveProc)( session->secshReceiveUserData,
						header + rcvd,
						(PGPInt32)( kPGPsecsh_RecordLengthSize - rcvd ) );
		if( bytesRead < 0 )
		{
			if( bytesRead != kPGPError_SECSHWouldBlock )
			{
				FATALSECSH( kPGPError_SECSHUnexpectedClose );
			}
			else
            {
                rerr = kPGPError_SECSHWouldBlock;
				bytesRead = 0;
            }
		}
		if( bytesRead == 0 )
		{
			if( session->blocking )
			{
				FATALSECSH( kPGPError_SECSHUnexpectedClose );
			}
			else
			{
				if( rcvd )
                {
                    err = pgpSECSHBufferRawData( session, header, rcvd );  CKERR;
                    if(rerr == kPGPError_SECSHWouldBlock)
                    /* check to see if receive call returned a blocking error */
                    {
                        err = kPGPError_SECSHWouldBlock;
                    }
                }
                else if(rerr == kPGPError_SECSHWouldBlock)
                /* check for blocking error */
                {
                    err = kPGPError_SECSHWouldBlock;
                }
				goto done;
			}
		}
	}
	pgpAssert( rcvd == kPGPsecsh_RecordLengthSize );

	length = PGPEndianToUInt32( kPGPBigEndian, &header[0] );
	if( length <= 0 || length > kPGPsecsh_MaximumPacketSize )
	{
		(void)pgpSECSHAlert( session,	kPGPsecsh_AL_FatalAlert,
									kPGPsecsh_AT_DecodeError );
		FATALSECSH( kPGPError_SECSHProtocolViolation );
	}
	padLength = 8 - (length & 7);
	bufLength = length + padLength;
	buffer = (PGPByte *) PGPNewData(session->memMgr, bufLength, 0 );
	CKNULL( buffer );
	rcvd = bufLength;
	err = pgpSECSHExtractRawData( session, buffer, &rcvd );	CKERR;
	for(; rcvd < bufLength; rcvd += bytesRead)
	{
		bytesRead = (session->secshReceiveProc)( session->secshReceiveUserData,
						buffer + rcvd, (PGPInt32)( bufLength - rcvd ) );
		if( bytesRead < 0 )
		{
			if( bytesRead != kPGPError_SECSHWouldBlock )
			{
				(void)PGPFreeData( buffer );
				buffer = NULL;
				FATALSECSH( kPGPError_SECSHUnexpectedClose );
			}
			else
            {
                bytesRead = 0;
                rerr = kPGPError_SECSHWouldBlock;  
            }
		}
		if( bytesRead == 0 )
		{
			if( session->blocking )
			{
				FATALSECSH( kPGPError_SECSHUnexpectedClose );
			}
			else
			{
				if( rcvd )
				{
					err = pgpSECSHBufferRawData( session, header,
												kPGPsecsh_RecordLengthSize );CKERR;
					err = pgpSECSHBufferRawData( session, buffer, rcvd );	CKERR;
				}
                if(rerr == kPGPError_SECSHWouldBlock)
                    err = kPGPError_SECSHWouldBlock;
				(void)PGPFreeData( buffer );
				buffer = NULL;
				goto done;
			}
		}
	}
	pgpAssert( rcvd == bufLength );

	/* Decrypt data if appropriate */
	if( session->encrypting )
	{
		if( PGPCFBContextRefIsValid( session->readCipherCFB ) )
		{
			PGPCFBDecrypt( session->readCipherCFB, buffer, bufLength, buffer );
		} else if( PGPCBCContextRefIsValid( session->readCipherCBC ) ) {
			PGPCBCDecrypt( session->readCipherCBC, buffer, bufLength, buffer );
		}
	}

	/* Verify CRC */
	sCalculateCRC( buffer, bufLength - kPGPsecsh_CRCSize, crcCheck );

	if( !pgpMemoryEqual( crcCheck, buffer + bufLength - kPGPsecsh_CRCSize,
						 kPGPsecsh_CRCSize ) )
	{
		(void)pgpSECSHAlert( session,	kPGPsecsh_AL_FatalAlert,
									kPGPsecsh_AT_BadRecordCRC );
		FATALSECSH( kPGPError_SECSHProtocolViolation );
	}

	*outBuffer  = (PGPByte *) PGPNewData(session->memMgr,
										 length - 1 - kPGPsecsh_CRCSize, 0 );
	CKNULL( *outBuffer );
	pgpCopyMemory( buffer + padLength + 1, *outBuffer, length - kPGPsecsh_CRCSize );
	
	*outType = buffer[padLength];
	*outLength	= length - 1 - kPGPsecsh_CRCSize;

done:
	if( IsntNull( buffer ) )
	{
		(void)PGPFreeData( buffer );
	}
	return err;
}


	PGPError
pgpSECSHReceiveServerVersion(
	PGPsecshSessionPriv *	session,
	PGPByte *				outMajor,
	PGPByte *				outMinor )
{
	PGPError				err	= kPGPError_NoErr;
    PGPError            	rerr = kPGPError_NoErr;
	PGPByte 				buffer[kPGPsecsh_MaximumIDStringSize];
	PGPByte *				bp0;
	PGPByte *				bp1;
	PGPByte *				bp2;
	PGPSize					rcvd;
	PGPByte					major;
	PGPByte					minor;
	PGPInt32				bytesRead;

	*outMajor	= 0;
	*outMinor	= 0;

	rcvd = sizeof(buffer) - 1;
	err = pgpSECSHExtractRawData( session, buffer, &rcvd );	CKERR;
	buffer[rcvd] = '\0';
	while( IsNull( bp0 = strchr( buffer, '\n' ) ) )
	{
		bytesRead = (session->secshReceiveProc)( session->secshReceiveUserData,
						buffer + rcvd,
						(PGPInt32)( sizeof(buffer) - 1 - rcvd ) );
		if( bytesRead < 0 )
		{
			if( bytesRead != kPGPError_SECSHWouldBlock )
			{
				FATALSECSH( kPGPError_SECSHUnexpectedClose );
			}
			else
            {
                rerr = kPGPError_SECSHWouldBlock;
				bytesRead = 0;
            }
		}
		if( bytesRead == 0 )
		{
			if( session->blocking )
			{
				FATALSECSH( kPGPError_SECSHUnexpectedClose );
			}
			else
			{
				if( rcvd )
                {
                    err = pgpSECSHBufferRawData( session, buffer, rcvd );  CKERR;
                    if(rerr == kPGPError_SECSHWouldBlock)
                    /* check to see if receive call returned a blocking error */
                    {
                        err = kPGPError_SECSHWouldBlock;
                    }
                }
                else if(rerr == kPGPError_SECSHWouldBlock)
                /* check for blocking error */
                {
                    err = kPGPError_SECSHWouldBlock;
                }
				goto done;
			}
		}
		rcvd += bytesRead;
		buffer[rcvd] = '\0';
	}

	if( !pgpMemoryEqual( buffer, SECSHIDPREFIX, strlen(SECSHIDPREFIX) )
		|| IsNull( bp2 = strchr( buffer, '.' ) ) )
	{
		(void)pgpSECSHAlert( session,	kPGPsecsh_AL_FatalAlert,
									kPGPsecsh_AT_IDFailure );
		FATALSECSH( kPGPError_SECSHProtocolViolation );
	}

	if( rcvd > (PGPSize)(bp0 - buffer + 1) )
	{
		err = pgpSECSHBufferRawData( session, bp0 + 1,
									 rcvd - (bp0 - buffer + 1) );  CKERR;
	}

	bp1 = buffer + strlen( SECSHIDPREFIX );
	major = atoi( bp1 );
	minor = atoi( bp2 + 1 );

	*outMajor	= major;
	*outMinor	= minor;
done:
	return err;
}


	PGPError
pgpSECSHSendQueueIdleInternal(
	PGPsecshSessionPriv *	session )
{
	PGPError			err = kPGPError_NoErr;
	PGPInt32			serr;
	
	if( IsntNull( session->queuedSendData ) && ( session->queuedSendSize > 0 ) )
	{
		serr = (session->secshSendProc)( session->secshSendUserData,
							session->queuedSendData,
							(PGPInt32) session->queuedSendSize );
		if( serr == kPGPError_SECSHWouldBlock )
		{
			err = kPGPError_SECSHWouldBlock;
		}
		else if( serr == (PGPInt32) session->queuedSendSize )
		{
			session->queuedSendSize = 0;
			err = PGPReallocData( session->memMgr,
					(void **) &session->queuedSendData, 0, 0 ); CKERR;
		}
		else
			FATALSECSH( kPGPError_SECSHUnexpectedClose );
	}
done:
	return err;
}


	PGPError
pgpSECSHSendPacket(
	PGPsecshSessionPriv *	session,
	PGPByte					pktType,
	const PGPByte *			inBuffer,
	PGPSize					inLength )
{
	PGPError				err	= kPGPError_NoErr;
	PGPByte *				buffer = NULL;
	PGPSize					bufLength = 0,
							padLength = 0,
							bufOff = 0;
	PGPInt32				serr = 0;
	
	PGPValidatePtr( inBuffer );
	if( inLength+1 > kPGPsecsh_MaximumPacketSize )
	{
		err = kPGPError_BufferTooSmall;
		goto done;				/* non-fatal */
	}

	padLength = 8 - ((inLength+5) & 7);
	bufLength = padLength + 1 + inLength + kPGPsecsh_CRCSize;
	bufOff = sizeof(PGPUInt32);		/* skip past length */
	buffer = (PGPByte *) PGPNewData(session->memMgr, bufLength+bufOff, 0 );
	CKNULL( buffer );

	/* Store length */
	PGPUInt32ToEndian( 1+inLength+kPGPsecsh_CRCSize, kPGPBigEndian, buffer );

	/* Store padding and packet type */
	PGPContextGetRandomBytes( session->pgpContext, buffer+bufOff, padLength );
	buffer[bufOff+padLength] = pktType;

	/* Store data */
	pgpCopyMemory( inBuffer, buffer+bufOff+padLength+1, inLength );

	sCalculateCRC( buffer+bufOff, bufLength - kPGPsecsh_CRCSize,
				   buffer + bufOff + bufLength - kPGPsecsh_CRCSize );


	if( session->encrypting )
	{
		if( PGPCFBContextRefIsValid( session->writeCipherCFB ) )
		{
			PGPCFBEncrypt( session->writeCipherCFB, buffer+bufOff,
						   bufLength, buffer+bufOff );
		} else if( PGPCBCContextRefIsValid( session->writeCipherCBC ) ) {
			PGPCBCEncrypt( session->writeCipherCBC, buffer+bufOff,
						   bufLength, buffer+bufOff );
		}
	}

	err = pgpSECSHSendQueueIdleInternal( session );
	if( err == kPGPError_SECSHWouldBlock )
		serr = err;
	
	if( IsntPGPError( err ) && ( bufLength > 0 ) )
	{
		if( serr != kPGPError_SECSHWouldBlock )
			serr = (session->secshSendProc)( session->secshSendUserData,
										buffer, (PGPInt32)(bufLength+bufOff) );
		if( serr == kPGPError_SECSHWouldBlock )
		{
			err = pgpSECSHBufferSendData( session, buffer, bufLength+bufOff ); CKERR;
			err = kPGPError_SECSHWouldBlock;
		}
		else if( (PGPSize)serr != bufLength+bufOff )
			FATALSECSH( kPGPError_SECSHUnexpectedClose );
	}
done:
	if( IsntNull( buffer ) )
		(void)PGPFreeData( buffer );
	return err;
}


	PGPError
pgpSECSHSendRawData(
	PGPsecshSessionPriv *	session,
	const PGPByte *			inBuffer,
	PGPSize					inLength )
{
	PGPError				err	= kPGPError_NoErr;
	PGPInt32				serr = 0;
	
	PGPValidatePtr( inBuffer );
	if( inLength > kPGPsecsh_MaximumPacketSize )
	{
		err = kPGPError_BufferTooSmall;
		goto done;				/* non-fatal */
	}

	err = pgpSECSHSendQueueIdleInternal( session );
	if( err == kPGPError_SECSHWouldBlock )
		serr = err;
	
	if( IsntPGPError( err ) && ( inLength > 0 ) )
	{
		if( serr != kPGPError_SECSHWouldBlock )
			serr = (session->secshSendProc)( session->secshSendUserData,
										inBuffer, (PGPInt32) inLength );
		if( serr == kPGPError_SECSHWouldBlock )
		{
			err = pgpSECSHBufferSendData( session, inBuffer, inLength ); CKERR;
			err = kPGPError_SECSHWouldBlock;
		}
		else if( (PGPSize)serr != inLength )
			FATALSECSH( kPGPError_SECSHUnexpectedClose );
	}
done:
	return err;
}


	PGPError
pgpSECSHSendSessionKey(
	PGPsecshSessionPriv *	session)
{
	PGPByte *				sessionKey = NULL;
	PGPCipherAlgorithm		cipher;
	PGPSize					sbufSize;
	PGPSize					lbufSize;
	PGPPublicKeyContextRef	smallContext;
	PGPPublicKeyContextRef	largeContext;
	PGPByte *				sBuffer = NULL;
	PGPByte *				lBuffer = NULL;
	PGPUInt32				lbufOff;
	PGPUInt16				lbufBits;
	PGPByte *				payload = NULL;
	PGPSize					payloadLen;
	PGPUInt32				payOff;
	PGPByte					iv[16];
	PGPInt32				i;
	PGPSymmetricCipherContextRef
							symContext;
	PGPError				err = kPGPError_NoErr;

	err = pgpSECSHClientChooseAlgorithm( session, &cipher ); CKERR;

	sessionKey = (PGPByte *)PGPNewSecureData( session->memMgr, kPGPsecsh_SessionKeySize, 0 );
	CKNULL( sessionKey );

	PGPContextGetRandomBytes( session->pgpContext, sessionKey, kPGPsecsh_SessionKeySize );

	PGPGetPublicKeyOperationSizes( session->remoteServerKeyContext, NULL,
								   &sbufSize, NULL );
	PGPGetPublicKeyOperationSizes( session->remoteHostKeyContext, NULL,
								   &lbufSize, NULL );
	if( sbufSize < lbufSize )
	{
		smallContext = session->remoteServerKeyContext;
		largeContext = session->remoteHostKeyContext;
	} else {
		PGPUInt32 t = sbufSize; sbufSize = lbufSize; lbufSize = t;
		smallContext = session->remoteHostKeyContext;
		largeContext = session->remoteServerKeyContext;
	}

	sBuffer = (PGPByte *)PGPNewData( session->memMgr, sbufSize, 0 ); CKNULL( sBuffer );
	lBuffer = (PGPByte *)PGPNewData( session->memMgr, lbufSize, 0 ); CKNULL( lBuffer );
	
	err = PGPPublicKeyEncrypt( smallContext, sessionKey, kPGPsecsh_SessionKeySize,
							   sBuffer, &sbufSize ); CKERR;
	err = PGPPublicKeyEncrypt( largeContext, sBuffer, sbufSize, lBuffer,
							   &lbufSize ); CKERR;
	lbufBits = sCountBits( lBuffer, lbufSize, &lbufOff );

	payloadLen = 1 + kPGPsecsh_CookieSize + sizeof(PGPUInt16)
		+ lbufSize-lbufOff + sizeof(PGPUInt32);
	payload = (PGPByte *)PGPNewData( session->memMgr, payloadLen, 0 ); CKNULL( payload );

	payOff = 0;

	payload[payOff++] = (cipher==kPGPCipherAlgorithm_IDEA) ? kPGPsecsh_CT_IDEA
														   : kPGPsecsh_CT_3DES;
	pgpCopyMemory( session->cookie, payload+payOff, kPGPsecsh_CookieSize );
	payOff += kPGPsecsh_CookieSize;
	PGPUInt16ToEndian( lbufBits, kPGPBigEndian, payload+payOff );
	payOff += sizeof( PGPUInt16 );
	pgpCopyMemory( lBuffer+lbufOff, payload+payOff, lbufSize-lbufOff );
	payOff += lbufSize - lbufOff;
	PGPUInt32ToEndian( session->localProtocolFlags, kPGPBigEndian, payload+payOff );
	payOff += sizeof( PGPUInt32 );
	pgpAssert( payOff == payloadLen );

	err = pgpSECSHSendPacket( session, kPGPsecsh_CMsg_SessionKey, payload, payloadLen ); CKERR;

	/* Done with host key encryption */
	(void) PGPFreePublicKeyContext( session->remoteHostKeyContext );
	(void) PGPFreePublicKeyContext( session->remoteServerKeyContext );
	session->remoteHostKeyContext   = kInvalidPGPPublicKeyContextRef;
	session->remoteServerKeyContext = kInvalidPGPPublicKeyContextRef;


	/* Set up IV and keys for symmetric encryption */
	pgpClearMemory( iv, sizeof( iv ) );
	for( i=0; i<kPGPsecsh_SessionIDSize; ++i )
	{
		sessionKey[i] ^= session->sessionID[i];
	}

	if( cipher == kPGPCipherAlgorithm_IDEA )
	{
		err = PGPNewSymmetricCipherContext( session->pgpContext,
											kPGPCipherAlgorithm_IDEA,
											&symContext ); CKERR;
		err = PGPNewCFBContext( symContext, 1, &session->writeCipherCFB ); CKERR;
		symContext = kInvalidPGPSymmetricCipherContextRef;
		err = PGPInitCFB( session->writeCipherCFB, sessionKey, iv ); CKERR;
		err = PGPNewSymmetricCipherContext( session->pgpContext,
											kPGPCipherAlgorithm_IDEA,
											&symContext ); CKERR;
		err = PGPNewCFBContext( symContext, 1, &session->readCipherCFB ); CKERR;
		symContext = kInvalidPGPSymmetricCipherContextRef;
		err = PGPInitCFB( session->readCipherCFB, sessionKey, iv ); CKERR;
	} else {
		pgpAssert( 0 );
	}

	session->encrypting = TRUE;


	
done:
	if( PGPSymmetricCipherContextRefIsValid( symContext ) )
		PGPFreeSymmetricCipherContext( symContext );
	if( IsntNull( sessionKey ) )
		PGPFreeData( sessionKey );
	if( IsntNull( payload ) )
		PGPFreeData( payload );
	if( IsntNull( sBuffer ) )
		PGPFreeData( sBuffer );
	if( IsntNull( lBuffer ) )
		PGPFreeData( lBuffer );

	return err;
}


	PGPError
pgpSECSHSendUserName(
	PGPsecshSessionPriv *	session)
{
	PGPByte *				payload = NULL;
	PGPUInt32				payOff = 0;
	PGPSize					payloadSize;
	PGPUInt32				nameLen;
	PGPError				err = kPGPError_NoErr;

	nameLen = strlen( session->userName );
	payloadSize = sizeof(PGPUInt32) + nameLen;

	payload = (PGPByte *)PGPNewData( session->memMgr, payloadSize, 0 );
	CKNULL( payload );

	PGPUInt32ToEndian( nameLen, kPGPBigEndian, payload+payOff );
	payOff += sizeof( PGPUInt32 );
	pgpCopyMemory( session->userName, payload+payOff, nameLen );
	payOff += nameLen;
	pgpAssert( payOff == payloadSize );
	
	err = pgpSECSHSendPacket( session, kPGPsecsh_CMsg_User, payload, payloadSize ); CKERR;

 done:
	if( IsntNull( payload ) )
		PGPFreeData( payload );
	return err;
}


	PGPError
pgpSECSHSendRSAChallengeRequest(
	PGPsecshSessionPriv *	session)
{
	void *					vbuf;
	PGPByte *				mbuf = NULL;
	PGPSize					mbufSize;
	PGPUInt16				mbufBits;
	PGPError				err = kPGPError_NoErr;

	/* Send modulus of local authentication key */
	err = PGPGetKeyDBObjAllocatedDataProperty( session->localKey,
											   kPGPKeyProperty_KeyData,
											   &vbuf, &mbufSize ); CKERR;
	mbuf = vbuf;
	mbufBits = (mbuf[0] << 8) | mbuf[1];
	mbufSize = 2 + ((mbufBits + 7) / 8);

	err = pgpSECSHSendPacket( session, kPGPsecsh_CMsg_AuthRSA, mbuf, mbufSize ); CKERR;

 done:
	if( IsntNull( mbuf ) )
		PGPFreeData( mbuf );
	return err;
}


	PGPError
pgpSECSHSendRSAResponse(
	PGPsecshSessionPriv *		session,
	const PGPByte *				inChallenge,
	PGPSize						inLength )
{
	PGPPrivateKeyContextRef		privKeyCon = kInvalidPGPPrivateKeyContextRef;
	PGPPublicKeyMessageFormat	format;
	PGPSize						bufSize;
	PGPByte *					decryptBuf = NULL;
	PGPByte *					payload = NULL;
	PGPHashContextRef			handshakeMD5 = kInvalidPGPHashContextRef;
	PGPError					err;
	
	format = kPGPPublicKeyMessageFormat_PKCS1;
	err = PGPNewPrivateKeyContext( PGPPeekKeyDBObjKey( session->localKey ),
		format, &privKeyCon, session->localKeyUsePasskey ?
		PGPOPasskeyBuffer( session->pgpContext, session->localKeyPasskeyBuffer,
							session->localKeyPasskeySize ) :
		PGPOPassphrase( session->pgpContext, session->localKeyPassphrase ),
		PGPOLastOption( session->pgpContext ) );	CKERR;

	err = PGPGetPrivateKeyOperationSizes( privKeyCon, &bufSize, NULL, NULL ); CKERR;
	decryptBuf = (PGPByte *)PGPNewData( session->memMgr, bufSize, 0 );
	CKNULL( decryptBuf );

	err = PGPPrivateKeyDecrypt( privKeyCon, inChallenge, inLength, decryptBuf,
								&bufSize ); CKERR;

	PGPFreePrivateKeyContext( privKeyCon );
	privKeyCon = kInvalidPGPPrivateKeyContextRef;

	if( bufSize > 32 )
	{
		(void)pgpSECSHAlert( session,	kPGPsecsh_AL_FatalAlert,
									kPGPsecsh_AT_DecodeError );
		FATALSECSH( kPGPError_SECSHProtocolViolation );
	}

	payload = (PGPByte *)PGPNewData( session->memMgr, kPGPsecsh_SessionIDSize, 0 );
	CKNULL( payload );

	err = PGPNewHashContext( session->pgpContext, kPGPHashAlgorithm_MD5,
							 &handshakeMD5 ); CKERR;
	
	PGPContinueHash( handshakeMD5, decryptBuf, bufSize );
	PGPContinueHash( handshakeMD5, session->sessionID, kPGPsecsh_SessionIDSize );

	PGPFinalizeHash( handshakeMD5, payload );
	PGPFreeHashContext( handshakeMD5 );
	handshakeMD5 = kInvalidPGPHashContextRef;

	err = pgpSECSHSendPacket( session, kPGPsecsh_CMsg_AuthRSAResponse,
							  payload, kPGPsecsh_SessionIDSize ); CKERR;

 done:
	if( IsntNull( payload ) )
		PGPFreeData( payload );
	if( IsntNull( decryptBuf ) )
		PGPFreeData( decryptBuf );
	if( PGPPrivateKeyContextRefIsValid( privKeyCon ) )
		PGPFreePrivateKeyContext( privKeyCon );
	if( PGPHashContextRefIsValid( handshakeMD5 ) )
		PGPFreeHashContext( handshakeMD5 );

	return err;
}


	PGPError
pgpSECSHReceivePublicKey(
	PGPsecshSessionPriv *			session,
	const PGPByte *					inBuffer,
	PGPSize							inLength )
{
	PGPError			err	= kPGPError_NoErr;
	PGPHashContextRef	handshakeMD5 = kInvalidPGPHashContextRef;
	PGPUInt16			pktInx = 0;
	PGPUInt32			sbits;
	PGPUInt32			hbits;
	PGPUInt16			sKeyOff;
	PGPUInt16			hKeyOff;
	PGPSize				ilen;

	/* First read cookie */
	if( (PGPSize)(pktInx + kPGPsecsh_CookieSize) > inLength )
	{
		(void)pgpSECSHAlert( session,	kPGPsecsh_AL_FatalAlert,
									kPGPsecsh_AT_DecodeError );
		FATALSECSH( kPGPError_SECSHProtocolViolation );
	}
	pgpCopyMemory( (PGPByte *)inBuffer+pktInx, session->cookie, kPGPsecsh_CookieSize );
	pktInx += kPGPsecsh_CookieSize;

	/* Server key bits */
	if( pktInx + sizeof(PGPUInt32) > inLength )
	{
		(void)pgpSECSHAlert( session,	kPGPsecsh_AL_FatalAlert,
									kPGPsecsh_AT_DecodeError );
		FATALSECSH( kPGPError_SECSHProtocolViolation );
	}
	sbits = PGPEndianToUInt32( kPGPBigEndian, inBuffer+pktInx );
	pktInx += sizeof(PGPUInt32);

	/* Server key data */
	err = sDataToPGPKey( session, (PGPByte *)inBuffer+pktInx, inLength-pktInx,
						 &ilen, &session->remoteServerKeyDB,
						 &session->remoteServerKey ); CKERR;
	pktInx += ilen;
	sKeyOff = pktInx - (sbits+7)/8;

	/* Host key bits */
	if( pktInx + sizeof(PGPUInt32) > inLength )
	{
		(void)pgpSECSHAlert( session,	kPGPsecsh_AL_FatalAlert,
									kPGPsecsh_AT_DecodeError );
		FATALSECSH( kPGPError_SECSHProtocolViolation );
	}
	hbits = PGPEndianToUInt32( kPGPBigEndian, inBuffer+pktInx );
	pktInx += sizeof(PGPUInt32);

	/* Server key data */
	err = sDataToPGPKey( session, (PGPByte *)inBuffer+pktInx, inLength-pktInx,
						 &ilen, &session->remoteHostKeyDB,
						 &session->remoteHostKey ); CKERR;
	pktInx += ilen;
	hKeyOff = pktInx - (hbits+7)/8;

	/* Protocol flags, cipher mask, auth mask */
	if( pktInx + 3*sizeof(PGPUInt32) > inLength )
	{
		(void)pgpSECSHAlert( session,	kPGPsecsh_AL_FatalAlert,
									kPGPsecsh_AT_DecodeError );
		FATALSECSH( kPGPError_SECSHProtocolViolation );
	}
	session->remoteProtocolFlags = PGPEndianToUInt32( kPGPBigEndian, inBuffer+pktInx);
	pktInx += sizeof(PGPUInt32);
	session->cipherMask = PGPEndianToUInt32( kPGPBigEndian, inBuffer+pktInx );
	pktInx += sizeof(PGPUInt32);
	session->authMask = PGPEndianToUInt32( kPGPBigEndian, inBuffer+pktInx );
	pktInx += sizeof(PGPUInt32);

	/* Calculate session ID */
	err = PGPNewHashContext( session->pgpContext, kPGPHashAlgorithm_MD5,
							 &handshakeMD5 ); CKERR;
	
	PGPContinueHash( handshakeMD5, inBuffer+hKeyOff, (hbits+7)/8 );
	PGPContinueHash( handshakeMD5, inBuffer+sKeyOff, (sbits+7)/8 );
	PGPContinueHash( handshakeMD5, session->cookie, kPGPsecsh_CookieSize );
	PGPFinalizeHash( handshakeMD5, session->sessionID );
	PGPFreeHashContext( handshakeMD5 );
	handshakeMD5 = kInvalidPGPHashContextRef;
	
	err = PGPNewPublicKeyContext( session->remoteServerKey,
								  kPGPPublicKeyMessageFormat_PKCS1,
								  &session->remoteServerKeyContext ); CKERR;
	err = PGPNewPublicKeyContext( session->remoteHostKey,
								  kPGPPublicKeyMessageFormat_PKCS1,
								  &session->remoteHostKeyContext ); CKERR;

	err = pgpSECSHSendSessionKey( session ); CKERR;

done:
	if( PGPHashContextRefIsValid( handshakeMD5 ) )
		PGPFreeHashContext( handshakeMD5 );
	return err;
}

	PGPError
pgpSECSHReceiveRSAChallenge(
	PGPsecshSessionPriv *			session,
	const PGPByte *					inBuffer,
	PGPSize							inLength )
{
	PGPError			err	= kPGPError_NoErr;
	PGPByte *			rsaChallenge = NULL;
	PGPUInt16			pktInx = 0;
	PGPUInt16			chalBits;
	PGPSize				chalSize;

	chalBits = (inBuffer[pktInx] << 8) + inBuffer[pktInx+1];
	chalSize = (chalBits + 7) / 8;
	pktInx += 2;

	if( inLength != (PGPSize) (chalSize + pktInx) )
	{
		(void)pgpSECSHAlert( session,	kPGPsecsh_AL_FatalAlert,
									kPGPsecsh_AT_DecodeError );
		FATALSECSH( kPGPError_SECSHProtocolViolation );
	}
	rsaChallenge = (PGPByte *)PGPNewData( session->memMgr, chalSize, 0 );
	CKNULL( rsaChallenge );

	pgpCopyMemory( inBuffer+pktInx, rsaChallenge, chalSize );

	err = pgpSECSHSendRSAResponse( session, rsaChallenge, chalSize );  CKERR;
		
 done:
	if( IsntNull( rsaChallenge ) )
		PGPFreeData( rsaChallenge );
	return err;
}


	PGPError
pgpSECSHReceiveDebug(
	PGPsecshSessionPriv *			session,
	const PGPByte *					inBuffer,
	PGPSize							inLength )
{
	PGPError			err	= kPGPError_NoErr;
	PGPUInt16			pktInx = 0;
	PGPUInt32			strCount;

	/* Read count of characters */
	strCount = PGPEndianToUInt32( kPGPBigEndian, inBuffer+pktInx );
	pktInx += sizeof(PGPUInt32);
	if( strCount > inLength - pktInx )
		strCount = inLength - pktInx;

#if PGP_DEBUG
	{
		char *str = PGPNewData( session->memMgr, strCount+1, 0 );
		pgpCopyMemory( inBuffer+pktInx, str, strCount );
		str[strCount] = 0;
		pgpDebugFmtMsg((pgpaFmtPrefix, "SECSH debug message: %s\n", str ));
		PGPFreeData( str );
	}
#endif
	
	return err;
}




/* Receive a success/failure message.  State has already changed. */
	PGPError
pgpSECSHReceiveSuccessFail(
	PGPsecshSessionPriv *			session,
	const PGPByte *					inBuffer,
	PGPSize							inLength )
{
	(void) inBuffer;
	(void) inLength;

	if( session->intState == 2 )
		pgpSECSHSendUserName( session );
	else if( session->intState == 3 )
		pgpSECSHSendRSAChallengeRequest( session );
	else
		pgpAssert( 0 );
	return kPGPError_NoErr;
}


	PGPError
pgpSECSHAlert(
	PGPsecshSessionPriv *	session,
	PGPByte					level,
	PGPsecshAlert			type )
{
	PGPError			err	= kPGPError_NoErr;
	
	if( level == kPGPsecsh_AL_FatalAlert )
	{
		session->state		= kPGPsecsh_FatalErrorState;
		session->fatalAlert	= type;
	}
	return err;
}



	PGPInt8
pgpSECSHPacketToEvent(
	PGPByte		packetType )
{
	PGPInt8		event = 0;
	
	switch( packetType )
	{
		case kPGPsecsh_SMsg_PublicKey:
			event = kPGPsecsh_EV_ReceiveServerPublicKey;
			break;
		case kPGPsecsh_SMsg_AuthRSAChallenge:
			event = kPGPsecsh_EV_ReceiveRSAChallenge;
			break;
		case kPGPsecsh_SMsg_Success:
			event = kPGPsecsh_EV_ReceiveSuccess;
			break;
		case kPGPsecsh_SMsg_Failure:
			event = kPGPsecsh_EV_ReceiveFailure;
			break;
		case kPGPsecsh_SMsg_Debug:
			event = kPGPsecsh_EV_ReceiveDebug;
			break;
		default:
			event = kPGPsecsh_EV_ReceiveUnknownPacket;
			break;
	}
	return event;
}


	PGPError
pgpSECSHClientHandshake(
	PGPsecshSessionPriv *	session )
{
	PGPError				err	= kPGPError_NoErr;
	PGPByte					pktType;
	PGPByte *				packet = NULL;
	PGPSize					pktLen;
	PGPInt8					newState;
	PGPByte					serverMajor;
	PGPByte					serverMinor;
	PGPBoolean				handshaking = TRUE;
	
	if( ( session->state == kPGPsecsh_IdleState ) || 
		( session->state == kPGPsecsh_ReadyState ) )
	{
		session->state = kPGPsecsh_HandshakeState;
		err = pgpSECSHReceiveServerVersion( session, &serverMajor,
											&serverMinor );
		if( IsPGPError( err ) )
			goto done;
		if( serverMajor > 1 )
		{
			(void)pgpSECSHAlert( session,	kPGPsecsh_AL_FatalAlert,
										kPGPsecsh_AT_UnsupportedVersion );
			FATALSECSH( kPGPError_SECSHProtocolViolation );
		}
		err = pgpSECSHSendRawData( session, SECSHID, strlen( SECSHID ) );
		if( IsPGPError( err ) )
			goto done;

	}
	while( handshaking &&
			IsntPGPError( err = pgpSECSHReceivePacket( session,
			&pktType, &packet, &pktLen ) ) && IsntNull( packet ) )
	{
		newState = pgpSECSHNextState( session, pgpSECSHPacketToEvent( pktType ) );
		if( newState == -2 )
		{
			(void)pgpSECSHAlert( session,	kPGPsecsh_AL_FatalAlert,
										kPGPsecsh_AT_UnexpectedMessage );
			FATALSECSH( kPGPError_SECSHProtocolViolation );
		}
		else if( newState == -1 )
		{
			/* do nothing */
		}
		else if( newState == SECSHNUMCLIENTSTATES )
		{
			session->state = kPGPsecsh_ReadyState;
			handshaking = FALSE;

		}
		else switch( pktType )
		{
			case kPGPsecsh_SMsg_PublicKey:
				err = pgpSECSHReceivePublicKey( session, packet, pktLen );
				break;
			case kPGPsecsh_SMsg_Success:
			case kPGPsecsh_SMsg_Failure:
				err = pgpSECSHReceiveSuccessFail( session, packet, pktLen );
				break;
			case kPGPsecsh_SMsg_AuthRSAChallenge:
				err = pgpSECSHReceiveRSAChallenge( session, packet, pktLen );
				break;
			case kPGPsecsh_SMsg_Debug:
				err = pgpSECSHReceiveDebug( session, packet, pktLen );
				break;
			default:
				pgpAssert( 0 );
		}
		(void)PGPFreeData( packet );
		packet = NULL;
		if( IsPGPError( err ) )
			goto done;
		if( !session->blocking )
			break;
	}
done:
	if( IsntNull( packet ) )
		(void)PGPFreeData( packet );
	return err;
}



	PGPError
PGPsecshHandshake(
	PGPsecshSessionRef		ref )
{
	PGPError				err	= kPGPError_NoErr;
	PGPsecshSessionPriv *	session;
	
	PGPValidatePtr( ref );

	session = (PGPsecshSessionPriv *) ref;
	PGPValidatePtr( session->secshReceiveProc );
	PGPValidatePtr( session->secshSendProc );

	if( ( session->state != kPGPsecsh_ReadyState ) &&
		( session->state != kPGPsecsh_IdleState ) &&
		( ( session->state != kPGPsecsh_HandshakeState ) ||
		session->blocking ) )
	{
		err = kPGPError_SECSHWrongState;
		goto done;
	}
	
/*
	if( session->isClientSide )
		err = pgpSECSHClientHandshake( session );
	else
		err = pgpSECSHServerHandshake( session );
*/
	err = pgpSECSHClientHandshake( session );
	
	if(err == kPGPError_SECSHWouldBlock)
		return kPGPError_NoErr;
	if( IsPGPError( err ) && session->state != kPGPsecsh_FatalErrorState )
		(void)pgpSECSHAlert( session,	kPGPsecsh_AL_FatalAlert,
									kPGPsecsh_AT_InternalError );
done:
	return err;
}


	PGPError
PGPsecshSetProtocolOptions(
	PGPsecshSessionRef		ref,
	PGPsecshFlags			options,
	PGPsecshProtocolFlags	pflags  )
{
	PGPsecshSessionPriv *	session;

	PGPValidatePtr( ref );
	
	session = (PGPsecshSessionPriv *) ref;
	
	if( options & kPGPsecshFlags_ServerSide )
		session->isClientSide = FALSE;
	if( options & kPGPsecshFlags_ClientSide )
		session->isClientSide = TRUE;
	if( options & kPGPsecshFlags_NonBlockingIO )
		session->blocking = FALSE;

	session->localProtocolFlags = (PGPUInt32) pflags;

	return kPGPError_NoErr;
}


static const PGPOptionType sSECSHOptionSet[] =
{
	kPGPOptionType_Passphrase,
	kPGPOptionType_Passkey
};


	PGPError
PGPsecshSetLocalPrivateKey(
	PGPsecshSessionRef		ref,
	char *					inUserName,
	PGPKeyDBObjRef			inKeyObject,
	char *					inHostName,
	PGPKeySetRef			inHostKeys,
	PGPOptionListRef		firstOption,
	... )
{
	PGPError				err	= kPGPError_NoErr;
	PGPsecshSessionPriv *		session;
	PGPBoolean				secret,
							disabled,
							revoked,
							expired,
							canDecrypt;
	PGPInt32				algID;
	void					*passedPhrase = NULL;
	PGPSize					passedLength;
	PGPOptionListRef		optionList = kInvalidPGPOptionListRef;
	va_list					args;
	PGPUInt32				userNameLen;
	PGPUInt32				hostNameLen;
	
	PGPValidatePtr( ref );
	PGPValidatePtr( inKeyObject );

	session = (PGPsecshSessionPriv *) ref;

	userNameLen = strlen( inUserName );
	session->userName = (char *)PGPNewData( session->memMgr, userNameLen, 0 );
	CKNULL( session->userName );
	pgpCopyMemory( inUserName, session->userName, userNameLen+1 );

	hostNameLen = strlen( inHostName );
	session->hostName = (char *)PGPNewData( session->memMgr, hostNameLen, 0 );
	CKNULL( session->hostName );
	pgpCopyMemory( inHostName, session->hostName, hostNameLen+1 );

	session->remoteHostLookupKeyset = inHostKeys;

	va_start( args, firstOption );
	optionList = pgpBuildOptionListArgs(session->pgpContext,
									FALSE, firstOption, args);
	va_end( args );
	
	pgpAssert( pgpOptionListIsValid( optionList ) );
	err = pgpGetOptionListError( optionList );	CKERR;
	err = pgpCheckOptionsInSet( optionList, sSECSHOptionSet,
				sizeof(sSECSHOptionSet) / sizeof(PGPOptionType) ); CKERR;
	
	session->localKey 	= inKeyObject;
	
	err = PGPGetKeyDBObjBooleanProperty( session->localKey, kPGPKeyProperty_IsSecret, &secret ); CKERR;
	err = PGPGetKeyDBObjBooleanProperty( session->localKey, kPGPKeyProperty_IsDisabled, &disabled ); CKERR;
	err = PGPGetKeyDBObjBooleanProperty( session->localKey, kPGPKeyProperty_IsExpired, &expired ); CKERR;
	err = PGPGetKeyDBObjBooleanProperty( session->localKey, kPGPKeyProperty_IsRevoked, &revoked ); CKERR;
	err = PGPGetKeyDBObjBooleanProperty( session->localKey, kPGPKeyProperty_CanDecrypt, &canDecrypt ); CKERR;
	
	if( !secret || disabled || expired || revoked || !canDecrypt )
	{
		err = kPGPError_SECSHKeyUnusable;
		goto done;
	}
	err = PGPGetKeyDBObjNumericProperty( session->localKey, kPGPKeyProperty_AlgorithmID, &algID ); CKERR;
	
	if( algID != kPGPPublicKeyAlgorithm_RSA )
	{
		err = kPGPError_SECSHKeyUnusable;
		goto done;
	}

	err = pgpFindOptionArgs( optionList, kPGPOptionType_Passphrase, FALSE,
						"%p%l", &passedPhrase, &passedLength ); CKERR;
	if( IsNull( passedPhrase ) )
	{
		err = pgpFindOptionArgs( optionList, kPGPOptionType_Passkey, FALSE,
						"%p%l", &passedPhrase, &passedLength ); CKERR;
		session->localKeyPasskeyBuffer = PGPNewSecureData( session->memMgr,
										passedLength,
										kPGPMemoryMgrFlags_Clear );
		if( IsNull( session->localKeyPasskeyBuffer ) )
		{
			err = kPGPError_OutOfMemory;
			goto done;
		}
		pgpCopyMemory( passedPhrase, session->localKeyPasskeyBuffer,
						passedLength );
		session->localKeyUsePasskey = TRUE;
	}
	else
	{
		session->localKeyPassphrase = PGPNewSecureData( session->memMgr,
										passedLength + 1,
										kPGPMemoryMgrFlags_Clear );
		if( IsNull( session->localKeyPassphrase ) )
		{
			err = kPGPError_OutOfMemory;
			goto done;
		}
		pgpCopyMemory( passedPhrase, session->localKeyPassphrase,
					passedLength );
		session->localKeyPassphrase[passedLength] = '\0';
		session->localKeyUsePasskey = FALSE;
	}
	
done:
	/* do not explicitly free passedPhrase, it is freed with the OptionList */
	if( PGPOptionListRefIsValid( optionList ) )
		PGPFreeOptionList( optionList );
	return err;
}
					

	PGPError
PGPsecshGetRemoteAuthenticatedKey(
	PGPsecshSessionRef	ref,
	PGPKeyDBObjRef *	outKey,
	PGPKeyDBRef *		outKeyDB )
{
	PGPError			err	= kPGPError_NoErr;
	PGPsecshSessionPriv *	session;

	PGPValidatePtr( ref );
	session = (PGPsecshSessionPriv *) ref;
	
	if( IsntNull( outKey ) )
		*outKey		= kInvalidPGPKeyDBObjRef;
	if( IsntNull( outKeyDB ) )
		*outKeyDB	= kInvalidPGPKeyDBRef;
	if( session->state == kPGPsecsh_ReadyState )
	{
		PGPKeyDBObjRef	hostKey = session->remoteHostKey;
		PGPKeyDBRef		hostDB = session->remoteHostKeyDB;

		/* See if key is present on incoming keyset */
		if( PGPKeySetRefIsValid( session->remoteHostLookupKeyset ) )
		{
			PGPByte			hostKeyHash[20];
			void *			vbuf;
			PGPSize			vbufSize;
			PGPKeyDBObjRef	matchKey;

			/* We look for key that matches in modulus/exponent */
			PGPGetKeyDBObjAllocatedDataProperty( hostKey,
						kPGPKeyProperty_KeyData, &vbuf, &vbufSize );
			pgpFingerprint20HashBuf( session->pgpContext, vbuf, vbufSize,
									 hostKeyHash );
			PGPFreeData( vbuf );
			matchKey = pgpKeyDBFindKey20n( PGPPeekKeySetKeyDB( session->remoteHostLookupKeyset),
										   hostKeyHash );
			if( PGPKeyDBObjRefIsValid( matchKey ) &&
				PGPKeySetIsMember( matchKey, session->remoteHostLookupKeyset ) )
			{
				hostKey = matchKey;
				hostDB = PGPPeekKeySetKeyDB( session->remoteHostLookupKeyset);
			}
		}

		if( IsntNull( outKey ) )
			*outKey		= hostKey;
		if( IsntNull( outKeyDB ) )
			*outKeyDB	= hostDB;
	}
	else
		err = kPGPError_SECSHWrongState;
	
	return err;
}


	PGPError
PGPsecshExportPublicKey(
	PGPKeyDBObjRef			key,
	char *					userName,
	char **					outBuffer,
	PGPSize *				outLength
	)
{
	PGPContextRef			context;
	PGPMemoryMgrRef			mgr;
	PGPBigNumRef			mod, exp;
	PGPUInt32				modbits, modlen;
	PGPUInt32				expbits, explen;
	PGPUInt32				alg;
	void *					vbuf;
	PGPByte *				dbuf;
	PGPSize					dbufLen;
	char *					buf;
	PGPSize					bufLen;
	PGPUInt32				bufOff;
	char					tbuf[20];
	PGPError				err = kPGPError_NoErr;

	PGPValidatePtr( key );
	PGPValidatePtr( userName );
	PGPValidatePtr( outBuffer );
	PGPValidatePtr( outLength );

	*outBuffer = NULL;
	*outLength = 0;

	err = PGPGetKeyDBObjNumericProperty( key,
						kPGPKeyProperty_AlgorithmID, &alg ); CKERR;
	if( alg != kPGPPublicKeyAlgorithm_RSA )
	{
		err = kPGPError_PublicKeyUnimplemented;
		goto done;
	}
	context = PGPPeekKeyDBObjContext( key );
	mgr = PGPPeekContextMemoryMgr( context );

	PGPNewBigNum( context, FALSE, &mod );
	PGPNewBigNum( context, FALSE, &exp );

	PGPGetKeyDBObjAllocatedDataProperty( key, kPGPKeyProperty_KeyData,
		&vbuf, &dbufLen );
	dbuf = vbuf;

	modbits = (dbuf[0]<<8) | dbuf[1];
	modlen = (modbits + 7) / 8;
	PGPBigNumInsertBigEndianBytes( mod, dbuf+2, 0, modlen );
	expbits = (dbuf[modlen+2]<<8) | dbuf[modlen+3];
	explen = (expbits + 7) / 8;
	PGPBigNumInsertBigEndianBytes( exp, dbuf+2+modlen+2, 0, explen );
	PGPFreeData( vbuf );

	bufLen = 0;
	bufLen += sprintf( tbuf, "%d ", modbits );
	bufLen += sPrint10( context, NULL, exp ) + 1;
	bufLen += sPrint10( context, NULL, mod ) + 1;
	bufLen += strlen(userName) + 1;

	buf = (PGPByte *)PGPNewData( mgr, bufLen, 0 );
	CKNULL( buf );

	bufOff = 0;
	bufOff += sprintf( buf+bufOff, "%d ", modbits );
	bufOff += sPrint10( context, buf+bufOff, exp );
	buf[bufOff++] = ' ';
	bufOff += sPrint10( context, buf+bufOff, mod );
	buf[bufOff++] = ' ';
	pgpCopyMemory( userName, buf+bufOff, strlen(userName) );
	bufOff += strlen(userName);
	buf[bufOff++] = '\0';
	pgpAssert( bufOff == bufLen );

	*outBuffer = buf;
	*outLength = bufLen - 1;		/* Don't count null */
	
	PGPFreeBigNum( mod );
	PGPFreeBigNum( exp );

 done:
	return err;
}



	PGPError
PGPNewSECSHContext(
	PGPContextRef			context,
	PGPsecshContextRef *	outRef )
{
	PGPError				err	= kPGPError_NoErr;
	PGPsecshContextPriv *	pContext;
	PGPMemoryMgrRef			memMgr;

	PGPValidatePtr( context );
	*outRef = NULL;
	
	memMgr = PGPPeekContextMemoryMgr( context );
	pContext = (PGPsecshContextPriv *) PGPNewData( memMgr,
											sizeof(PGPsecshContextPriv),
											kPGPMemoryMgrFlags_Clear );
	if( IsntNull( pContext ) )
	{
		pContext->pgpContext = context;
		*outRef = ( PGPsecshContextRef ) pContext;
	}
	else
		err = kPGPError_OutOfMemory;
	
	pgpAssertErrWithPtr( err, *outRef );
	return err;
}

	PGPError
PGPFreeSECSHContext(
	PGPsecshContextRef	ref )
{
	PGPValidatePtr( ref );

	return PGPFreeData(ref );
}

	PGPError
PGPNewSECSHSession(
	PGPsecshContextRef		ref,
	PGPsecshSessionRef *	outRef )
{
	PGPError				err	= kPGPError_NoErr;
	PGPsecshContextPriv *	pContext;
	PGPsecshSessionPriv *	session;
	PGPMemoryMgrRef			memMgr;

	PGPValidatePtr( ref );
	*outRef = NULL;
	
	pContext = ( PGPsecshContextPriv * ) ref;
	memMgr = PGPPeekContextMemoryMgr( pContext->pgpContext );
	session = ( PGPsecshSessionPriv * ) PGPNewData( memMgr,
											sizeof( PGPsecshSessionPriv ),
											kPGPMemoryMgrFlags_Clear );
	if( IsntNull( session ) )
	{
		session->pgpContext			= pContext->pgpContext;
		session->secshContext		= pContext;
		session->memMgr				= memMgr;
		session->state				= kPGPsecsh_IdleState;
		session->intState			= 0;
		session->isClientSide		= TRUE;
		session->blocking			= TRUE;
		session->encrypting			= FALSE;

		session->writeCipherCBC		= kInvalidPGPCBCContextRef;
		session->readCipherCBC		= kInvalidPGPCBCContextRef;
		session->writeCipherCFB		= kInvalidPGPCFBContextRef;
		session->readCipherCFB		= kInvalidPGPCFBContextRef;
		
		session->secshReceiveProc	= NULL;
		session->secshSendProc		= NULL;
		session->secshReceiveUserData = NULL;
		session->secshSendUserData	= NULL;

		session->fatalAlert			= kPGPsecsh_AT_None;
		
		*outRef = ( PGPsecshSessionRef ) session;
	}
	else
		err = kPGPError_OutOfMemory;
	
	if( IsPGPError( err ) && IsntNull( session ) )
	{
		PGPFreeData( session );
		*outRef = NULL;
	}
	pgpAssertErrWithPtr( err, *outRef );
	return err;
}


	PGPError
PGPFreeSECSHSession( PGPsecshSessionRef ref )
{
	PGPError			err	= kPGPError_NoErr;
	PGPsecshSessionPriv *	session;

	PGPValidatePtr( ref );

	session = (PGPsecshSessionPriv *) ref;
	
	if( IsntNull( session->localKeyPassphrase ) )
		(void)PGPFreeData( session->localKeyPassphrase );
	if( IsntNull( session->localKeyPasskeyBuffer ) )
		(void)PGPFreeData( session->localKeyPasskeyBuffer );
	if( IsntNull( session->userName ) )
		(void)PGPFreeData( session->userName );
	if( IsntNull( session->hostName ) )
		(void)PGPFreeData( session->hostName );
	if( PGPKeyDBRefIsValid( session->remoteHostKeyDB ) )
		(void)PGPFreeKeyDB( session->remoteHostKeyDB );
	if( PGPKeyDBRefIsValid( session->remoteServerKeyDB ) )
		(void)PGPFreeKeyDB( session->remoteServerKeyDB );
	
	if( PGPCBCContextRefIsValid( session->writeCipherCBC ) )
		(void)PGPFreeCBCContext( session->writeCipherCBC );
	if( PGPCBCContextRefIsValid( session->readCipherCBC ) )
		(void)PGPFreeCBCContext( session->readCipherCBC );
	if( PGPCFBContextRefIsValid( session->writeCipherCFB ) )
		(void)PGPFreeCFBContext( session->writeCipherCFB );
	if( PGPCFBContextRefIsValid( session->readCipherCFB ) )
		(void)PGPFreeCFBContext( session->readCipherCFB );
		
	if( IsntNull( session->queuedSendData ) )
		(void)PGPFreeData( session->queuedSendData );
	if( IsntNull( session->rcvdRawData ) )
		(void)PGPFreeData( session->rcvdRawData );
	

	err = PGPFreeData( session );
	return err;
}


	PGPError
PGPsecshClose(
	PGPsecshSessionRef		ref,
	PGPBoolean				dontCache )
{
	PGPError				err	= kPGPError_NoErr;
	PGPsecshSessionPriv *	session;
	
	PGPValidatePtr( ref );

	session = (PGPsecshSessionPriv *) ref;
	
	if( session->state == kPGPsecsh_ReadyState )
	{
		PGPByte closePkt[] = { 0, 0, 0, 0 };
		(void) pgpSECSHSendPacket( session, kPGPsecsh_Msg_Disconnect,
								   closePkt, sizeof(closePkt) );
	}
	if( session->state != kPGPsecsh_FatalErrorState )
		session->state = kPGPsecsh_ClosedState;
	
	return err;
}


	PGPError
PGPsecshSetReceiveCallback(
	PGPsecshSessionRef		ref,
	PGPsecshReceiveProcPtr	secshReceiveProc,
	void *					inData )
{
	PGPsecshSessionPriv *	session;

	PGPValidatePtr( ref );
	PGPValidatePtr( secshReceiveProc );
	
	session = (PGPsecshSessionPriv *) ref;
	session->secshReceiveProc		=	secshReceiveProc;
	session->secshReceiveUserData	=	inData;
	
	return kPGPError_NoErr;
}

	PGPError
PGPsecshSetSendCallback(
	PGPsecshSessionRef		ref,
	PGPsecshSendProcPtr		secshSendProc,
	void *					inData )
{
	PGPsecshSessionPriv *	session;

	PGPValidatePtr( ref );
	PGPValidatePtr( secshSendProc );
	
	session = (PGPsecshSessionPriv *) ref;
	session->secshSendProc		=	secshSendProc;
	session->secshSendUserData	=	inData;
	
	return kPGPError_NoErr;
}


	PGPError
PGPsecshReceive(
	PGPsecshSessionRef		ref,
	PGPByte *				outType,
	void **					outBuffer,
	PGPSize *				bufferSize )
{
	PGPError				err	= kPGPError_NoErr;
	PGPsecshSessionPriv *	session;
	PGPByte					pktType;
	PGPByte *				buffer = NULL;
	PGPSize					pktLen;

	PGPValidatePtr( ref );
	PGPValidatePtr( outBuffer );
	
	session = (PGPsecshSessionPriv *) ref;
	PGPValidatePtr( session->secshReceiveProc );

	*outType	= 0;
	*outBuffer	= NULL;
	*bufferSize	= 0;
	
	if( session->state != kPGPsecsh_ReadyState )
	{
		err = kPGPError_SECSHWrongState;
		goto done;
	}

	err = pgpSECSHReceivePacket( session, &pktType, &buffer, &pktLen ); CKERR;

	*outType = pktType;
	*outBuffer = buffer;
	*bufferSize = pktLen;

done:

	return err;
}

	PGPError
PGPsecshSend(
	PGPsecshSessionRef		ref,
	PGPByte					pktType,
	const void *			inBuffer,
	PGPSize					inBufferLength )
{
	PGPError				err	= kPGPError_NoErr;
	PGPsecshSessionPriv *	session;

	PGPValidatePtr( ref );
	PGPValidatePtr( inBuffer );

	session = (PGPsecshSessionPriv *) ref;
	PGPValidatePtr( session->secshSendProc );
	
	if( session->state != kPGPsecsh_ReadyState )
	{
		err = kPGPError_SECSHWrongState;
		goto done;
	}
	
	err = pgpSECSHSendPacket( session, pktType,
							  (PGPByte *)inBuffer, inBufferLength );

done:
	return err;
}




/*__Editor_settings____

	Local Variables:
	tab-width: 4
	End:
	vi: ts=4 sw=4
	vim: si
_____________________*/
